/*===========================================================================



SI-8B Geiger counter
====================



Overview:
=========

CPU         :   Arduino Nano / ATmega328
Clock       :   XTAL 16MHz
Compiler    :   Arduino 1.8.13
Version     :   1.02
Date        :   2022-10-27
Author      :   Iacopo Giangrandi HB9DUL



Description:
============

This program implements the counting functions and unit conversion of an 
SI-8B based Geiger counter. It accepts pulses on D2. Only D2 or D3 can be 
used for this, because they directly generate interrupt on change INT0 and 
INT1 respectively. Every time a pulse (falling edge) is detected, the 
counter is incremented by one. D3 is used to generate an interrupt every 
time the touch screen is touched.

Results are displayed on a 320x240 TFT screen and the user controls the 
unit with the touch screen. By clocking on the unit one can change the unit 
of the displayed result between uR/h, nSv/h and cpm. 

By clicking on the integration one, one can change it between 2, 10 and 60 
seconds. This software implements 60 different counters, each one 
integrating for one second. By changing the integration time between 2, 10 
and 60 seconds, one only changes the number of counters that are added 
together to form a running average. In this way, even with the longer 
integration time, one still gets a result per second, even if it take 60 
seconds to settle out.

The dose is not computed by considering the dead time of the tube. Instead, 
a cpm to uR/h curve supplied by the manufacturer is used. This curve has 
been digitized and approximated with 24 linear segments in the 
calculate_uR_h() function. 

The battery voltage is displayed on the top right corner of the screen. It 
flashes red when the battery is low. By clicking on it, it's possible to 
dim the display. Battery voltage is sampled via a voltage divider on pin 
D14/A0. A diode connects the battery to the Vdd rail, while the power can 
also come from the Arduino USB connector so, when powered via the USB, it 
will read 0.

Every time an option is changed, it's automatically saved in the EEPROM. I 
know that the write-life of the EEPROM is limited, but I think it should
last long enough. If not, it's till possible to change the address of the
variable and start gain for a new full life of 100'000 cycles.

My initials are drawn on the bottom right corner of the screen. By clicking 
on them one can display the "about" screen. While the about screen is 
displayed, no more data is sent on the serial port. It willy stay onscreen
for 10 seconds or until the screen is touched again.

Every second the display is updated and the raw measurements are also echoed 
on the serial port at 115200bps. As a reminder, the SI-8B tube produces 
350-500cps at 1uR/s. Dead time is 160us at 390V, but it's not used for 
computing the dose.



I/O connections:
================

SPI connections (LCD, TP, SD):
------------------------------
D11    ----> SPI_MOSI   SPI Master Output Slave Input line
D12    <---- SPI_MISO   SPI Master Input Slave Output line
D13    ----> SPI_SCK    SPI Clock line

LCD connections:
----------------
D6     ----> LCD_LED    LCD back-light (1=on)
D8     ----> LCD_RST    LCD reset line (0=reset, 1=operate) *
D9     ----> LCD_DC_RS  LCD dcrs line (1=register, 0=data)
D10    ----- LCD_CS     LCD chip select line (0=enable)

D5     ----> TS_CS      Touch Screen chip select line (0=enable)
D3     <---- TS_IRQ     Touch Screen IRQ line (1=idle, 0=IRQ) **

D7     ----> SD_CS      SD card Chip Select line (0=enable) **

UART connections:
-----------------
D0/RXD <---- USB_TX     UART communication (from PC to Arduino)
D1/TXD ----> USB_RX     UART communication (from Arduino to PC)

I/O connections:
----------------
D2     <---- IN_TUBE
D4     ----> TICK_EN    Geiger "Tick" enable line (1=tick, 0=silent)

D14/A0 <---- Battery voltage (analog) 100k/18k

*  Optional, but implemented in this firmware
** Connected but not implemented in the firmware



============================================================================= */



// --------------------------------------------------------------------------
// External files inclusions.
// --------------------------------------------------------------------------
#include <string.h>                                             // Used to format data before printing it on the screen
#include "TimerOne.h"                                           // Timer one is needed to periodically run timer1_interrupt()
#include <SPI.h>                                                // SPI is used for the display and the touch screen
#include <Adafruit_GFX.h>                                       // Core graphics library
#include <Adafruit_ILI9341.h>                                   // Fast hardware-specific library
#include <XPT2046_Touchscreen.h>                                // Needed for the touch screen
#include <EEPROM.h>                                             // Library to handle the EEPROM.



// --------------------------------------------------------------------------
// General definitions, constants and global variables.
// --------------------------------------------------------------------------
#define       WELCOME_MSG_1     "SI-8B Geiger Counter"          // Message shown on the display each time the device is powered up (4 lines).
#define       WELCOME_MSG_2     "v1.02 2022-10-27"
#define       WELCOME_MSG_3     "(c) Iacopo Giangrandi"
#define       WELCOME_MSG_4     "USB serial @ 115200N81"

#define       NSV_H_ORANGE      600                             // Low doses are displayed in green. Above this threshold (in nSv/h) the dose is displayed in orange.
#define       NSV_H_RED         2000                            // High doses (above this threshold in nSv/h) are displayed in red.
#define       UR_H_ORANGE       70                              // Same as above, but for uR/h. The equivalent of 600nSv/h is 70uR/h.
#define       UR_H_RED          250                             // The equivalent of 2000nSv/h is 250uR/h, but is rounded to 1600cpm.
#define       CPM_ORANGE        500                             // Same as above, but for cpm. The equivalent of 600nSv/h is 480cpm, but is rounded to 500cpm to line up with the analog meter scale.
#define       CPM_RED           1500                            // The equivalent of 2000nSv/h is 1570cpm, but is rounded to 1500cpm.

#define       TIME_SLOW         60                              // Long integration time is set to 60 seconds.
#define       TIME_MED          10                              // Medium integration time set to 10 seconds.
#define       TIME_FAST         2                               // Short long integration time set to 2 seconds.
char time_interval = TIME_MED;                                  // This variable defines the current integration time. Here we define the default integration time when the device is powered on.

#define       TIMER_PERIOD      1000000                         // Delay in us for the average to be updated. Here 1s.
#define       NUM_COUNTERS      61                              // This constant specifies how many counters to keep for the running average function. You needs one conter more than the number you are integrating, because a partial count yields to a wrong result.
volatile unsigned int counters[NUM_COUNTERS];                   // This is the actual array of counters to compute the running average.
char num_counters = (TIMER_PERIOD * time_interval / 1000000L) + 1;  // This is the actual number of counters in use. It can be smaller than NUM_COUNTERS, when averaging on faster periods.
volatile char current_counter = 0;                              // This is the index of the counter currently being incremented.
volatile char max_counter = 1;                                  // This is used to compute the average while the array is not filled yet: it is incremented from 1 to num_counters as the running average grows and then stays there.
#define       DATA_NOT_READY    0                               // These are the values that data_updated can take.
#define       DATA_READY        1
volatile char data_updated = DATA_NOT_READY;                    // This flag is updated by the timer1 interrupt every time a new average is available.

#define       UNIT_CPM          0                               // With this unit, counts per minute (cpm) are displayed.
#define       UNIT_R            1                               // With this unit, Roentgens per hour (R/h) are displayed.
#define       UNIT_SV           2                               // With this unit, Sieverts per hour (Sv/h) are displayed.
char unit_type = UNIT_R;                                        // This variable defines the current unit. Here we define the default unit when the device is powered on.

#define       TICKS_ENABLED     1                               // When tube_ticks is set to TICKS_ENABLED the speaker will tick at each pulse.
#define       TICKS_DISABLED    0                               // When tube_ticks is set to TICKS_DISABLED the speaker will not sound.
char tube_ticks = TICKS_ENABLED;                                // This variable defines if the speaker will sound or not.

int volt_batt;                                                  // This variable contains the last battery voltage reading in mV.
unsigned long int cpm_current, uR_h_current, nSv_h_current;     // These variable contain the last computed doses in the respective units.

#define       DISPLAY_BRIGHT    1                               // When display_brightness is set to DISPLAY_BRIGHT the LED is fully on.
#define       DISPLAY_DIM       0                               // When display_brightness is set to TICKS_DISABLED the LED is PWMed down.
char display_brightness = DISPLAY_BRIGHT;                       // This variable defines if the display is bright or not.
#define       DISPLAY_DIM_VALUE 63                              // This defines the duty cycle of the dimmed down display 0..255.

#define       TS_MIN_X          420                             // Touchscreen calibration: value when hitting the rightmost edge (x axis is reversed)
#define       TS_MAX_X          3900                            // Touchscreen calibration: value when hitting the leftmost edge (x axis is reversed)
#define       TS_MIN_Y          265                             // Touchscreen calibration: value when hitting the topmost edge
#define       TS_MAX_Y          3840                            // Touchscreen calibration: value when hitting the bottom-most edge

#define       CENTER_X          160                             // These are the coordinates of the center of rotation of the needle.
#define       CENTER_Y          360                             // As in many analog gauges it's located in the middle and just below (outside) the face of the instrument.
#define       RADIUS            295                             // This is the length of the arc drawing the scale.
#define       TICK_LENGTH       8                               // Short ticks are 8 pixels long.
#define       TICK_LENGTH_LONG  16                              // Long ticks are 16 pixels long.
#define       ANGLE             0.5                             // This defines the angular extension of the scale. Each unit (there are 100, form 0 to 100) is 0.5 degrees.
//#define       PI                3.1415926535897932384626433832795   // PI is already defined: no need to redefine it.

#define       EE_INIT_VALUE     0x78                            // Any value different from 0x00 and 0xFF will do. It's used to test if the EEPROM has been written once or not.
#define       EE_ADDR_EEINIT    0                               // EEPROM address of the EEPROM init check (EEPROM starts at 0).
#define       EE_ADDR_TIME_INT  1                               // EEPROM address of char time_interval
#define       EE_ADDR_UNIT_TYPE 2                               // EEPROM address of char unit_type
#define       EE_ADDR_TUBE_TKS  3                               // EEPROM address of char tube_ticks
#define       EE_ADDR_DISP_BRT  4                               // EEPROM address of char display_brightness



// --------------------------------------------------------------------------
// Hardware related definitions, constants and global variables.
// --------------------------------------------------------------------------
#define       IN_TUBE           2                               // Input for pulses of the tube. Must be on D2 to use INT0. Triggers on falling edge.
#define       VOLT_BATT         14                              // Analog input for battery rail (through voltage divider).
#define       TICK_EN           4

#define       VBATT_MIN         5000                            // The 3x AA batteries are considered low if below 5.0V
#define       VBATT_ZERO        100                             // But if the battery voltage is below 0.1V, we have no battery and we are running on USB power: non need to warn.

#define       CYCLE_DELAY       100                             // Time to wait at the end of each cycle, 100ms.

#define       SPI_MOSI          11                              // This is just a reminder: SPI pins are hardwired inside the MCU and they can only be pin 11, 12 and 13.
#define       SPI_MISO          12                              // These definitions are not used by this code, as the SPI interface can only use those lines.
#define       SPI_CLK           13

#define       LCD_LED           6                               // TFT control lines.
#define       LCD_RST           8
#define       LCD_DC_RS         9
#define       LCD_CS            10
Adafruit_ILI9341 tft = Adafruit_ILI9341(LCD_CS, LCD_DC_RS, LCD_RST);    // Invoke the TFT library
#define       TFT_BLACK         0x0000                          // A list of predefined colors can be handy.
#define       TFT_WHITE         0xFFFF                          // Bits 0..4 -> Blue 0..4; Bits 5..10 -> Green 0..5; Bits 11..15 -> Red 0..4
#define       TFT_GRAY          0x7BEF
#define       TFT_LIGHT_GRAY    0xC618
#define       TFT_GREEN         0x07E0
#define       TFT_LIME          0x87E0
#define       TFT_BLUE          0x001F
#define       TFT_RED           0xF800
#define       TFT_AQUA          0x5D1C
#define       TFT_YELLOW        0xFFE0
#define       TFT_MAGENTA       0xF81F
#define       TFT_CYAN          0x07FF
#define       TFT_DARK_CYAN     0x03EF
#define       TFT_ORANGE        0xFCA0
#define       TFT_PINK          0xF97F
#define       TFT_BROWN         0x8200
#define       TFT_VIOLET        0x9199
#define       TFT_SILVER        0xA510
#define       TFT_GOLD          0xA508
#define       TFT_NAVY          0x000F
#define       TFT_MAROON        0x7800
#define       TFT_PURPLE        0x780F
#define       TFT_OLIVE         0x7BE0
#define       TFT_DARK_GREEN    0x03E0

#define       TS_CS             5                               // Touch screen control lines.
#define       TS_IRQ            3
XPT2046_Touchscreen ts(TS_CS);
//XPT2046_Touchscreen ts(TS_CS, TS_IRQ);                          // We use it without IRQ, it seems more stable (the display freezes sometimes)

#define       SD_CS             7                               // SD-card (not used) control line.



// --------------------------------------------------------------------------
// This is the setup procedure executed once at startup. This procedure is
// automatically executed by Arduino.
// --------------------------------------------------------------------------
void setup()
{
    char i;
    
    // Read EEPROM configuration.

    eeprom_read_config();

    // Configure I/O lines.

    pinMode(IN_TUBE, INPUT_PULLUP);                             // Configure tubes pulses pins as inputs.
    attachInterrupt(0, int_tube, FALLING);                      // Enable interrupt on change for D2.

    pinMode(TICK_EN, OUTPUT);                                   // This is the tick enable line, it's an output.
    if (tube_ticks == TICKS_ENABLED)
        digitalWrite(TICK_EN, HIGH);                            // And set as specified in tube_ticks.
    else
        digitalWrite(TICK_EN, LOW);

    //pinMode(LCD_DC_RS, OUTPUT);                                 // LCD I/O lines are initialized by the TFT library.
    //digitalWrite(LCD_DC_RS, HIGH);
    //pinMode(LCD_CS, OUTPUT);
    //digitalWrite(LCD_CS, HIGH);
    //pinMode(LCD_RST, OUTPUT);
    //digitalWrite(LCD_RST, HIGH);
    pinMode(LCD_LED, OUTPUT);                                   // Configure the TFT LED line as an output.
    digitalWrite(LCD_LED, LOW);                                 // But keep it off for the moment as the display flickers when initialized. It will be switched on by display_welcome_screen() at the end of this init procedure.
    //set_tft_brightness();                                       // Set LED brightness according to display_brightness.

    //pinMode(TS_CS, OUTPUT);                                     // Touch screen I/O lines are initialized by its library.
    //digitalWrite(TP_CS, HIGH);
    pinMode(TS_IRQ, INPUT_PULLUP);

    pinMode(SD_CS, OUTPUT);                                     // SD-card is not used, just configure its CS line as an output.
    digitalWrite(SD_CS, HIGH);                                  // And disable it.

    analogReference(INTERNAL);                                  // Use the internal 1.1V analog reference.
    pinMode(VOLT_BATT, INPUT);                                  // Voltage input on A0(D14) to read battery voltage.

    // Configure serial port.

    Serial.begin(115200);                                       // Set serial communication at 115200N81.

    // Configure timer one for computing the running average

    for (i=0; i<NUM_COUNTERS; i++)                              // Clear all counters;
        counters[i] = 0;
    current_counter = 0;
    max_counter = 1;
    Timer1.initialize(TIMER_PERIOD);                            // Initialize timer1, and set a 1-second period
    Timer1.attachInterrupt(timer1_interrupt);                   // Attach timer1_interrupt() as a timer overflow interrupt

    // Initialize the TFT (LCD) display and touch screen

    tft.begin();                                                // Display initialization
    tft.setRotation(1);                                         // Set correct orientation (contacts on the right).

    ts.begin();                                                 // Touchscreen initialization
    ts.setRotation(1);                                          // Set correct orientation (contacts on the right).

    // Now we are done with initialization, we can show the welcome message.

    display_welcome_screen(false);                              // Start with the usual greetings and don't wait for the touch screen. This will also switch on the display back-light.
}




// --------------------------------------------------------------------------
// This is the main program. It's executed automatically after the 
// initialization in an endless loop by Arduino.
// --------------------------------------------------------------------------
void loop()
{
    char i;                                                     // General purpose counter.

    if (data_updated == DATA_READY)                             // Check if integration time is elapsed.
    {
        data_updated = DATA_NOT_READY;                          // Clear flag, we're now processing the data.

        cpm_current = 0;                                        // Sum all counters together (except the one being currently incremented) to get the average.
        for (i=0; i<max_counter; i++)                           // max_counter can be incremented in the background by the interrupt, but if this happens it's ok: it only means that there is an additional counter to add. And there is no problem for the loop test as max_counter is an 8-bit variable.
        {
            unsigned int counter_tmp;                           // Temporary variable used to minimize the time interrupts are disabled.

            noInterrupts();                                     // Prevent other interrupts from happening while we manipulate the counters as this requires several 8-bit memory operations.
            if (i != current_counter)                           // Skip the counter currently being incremented.
                counter_tmp = counters[i];                      // If not, save it on a temporary variable...
            else
                counter_tmp = 0;
            interrupts();                                       // ...and enable interrupts again as we're done with the critical stuff.
            cpm_current += counter_tmp;
        }
        i--;                                                    // When the loop ends i is equal to the total number of counters effectively added plus the one that we skipped, usually equals to max_counter. But even if max_counter is incremented by the interrupt i still has the correct number of counters. Now decrease i by 1 to get the total number of added counters.
        cpm_current *= 60 * 100;                                // Compute counts per minute without overflowing.
        cpm_current /= TIMER_PERIOD / 10000;                    // TIMER_PERIOD is in us, the factors 100 (above) and 10'000 (here) convert it into seconds.
        cpm_current /= i;                                       // Average them by dividing by the actual number of counters in use (this number starts at 1 and grows up to num_counters as the average runs). Here, instead of dividing by max_counter we divide by i. This is because max_counter can be incremented by the interrupt at any time. On the other hand, after the previous loop i contains the total number of counters that have been added together, regardless if max_counter was then incremented in the background or not.
        uR_h_current = calculate_uR_h(cpm_current);             // Now convert CPM into uR/h
        nSv_h_current = calculate_nSv_h(uR_h_current);          // And finally uR/h into nSv/h

        display_dose();                                         // Now we are ready to display the dose on the screen.
        display_gauge(false);

        Serial.print(TIMER_PERIOD/1000);                        // Communicate result on the serial port. Convert TIMER_PERIOD into ms.
        Serial.print(';');
        if (current_counter > 0)                                // Echo the last updated counter on the serial port (the one that precedes the one being incremented right now).
            Serial.print(counters[current_counter-1]);          // As this counter is never written by the interrupts (remember, it's the previous one) we can read it without disabling the interrupts.
        else
            Serial.print(counters[num_counters-1]);
        Serial.print(';');
        Serial.print(cpm_current);
        Serial.print(';');
        Serial.print(uR_h_current);
        Serial.print(';');
        Serial.print(nSv_h_current);
        Serial.print(';');
        Serial.print(int(num_counters-1));
        Serial.print(';');
        Serial.println(volt_batt);
    }

    touch_panel_handler();                                      // Handle user input on the touch screen.

    display_integration_time();                                 // Display the integration time (regardless if it was changed by the user or not).
    display_sound_status();                                     // Display speaker status.
    display_my_initials();

    volt_batt = read_battery_voltage();                         // Read the battery voltages.
    display_voltage(volt_batt);

    delay(CYCLE_DELAY);                                         // Wait some time and cycle again (and again).
}



// --------------------------------------------------------------------------
// Interrupt service routine (INT0) run each time the tube generates a pulse.
// --------------------------------------------------------------------------
void int_tube()
{
    counters[current_counter]++;                                // Simply increments the corresponding counter.
}



// --------------------------------------------------------------------------
// Interrupt service routine (TMR1) run once per second.
// --------------------------------------------------------------------------
void timer1_interrupt()
{
    current_counter++;                                          // Increment the counter and handle overflow.
    if (current_counter >= num_counters)
        current_counter = 0;
    counters[current_counter] = 0;                              // And clear the next counter so that we start at zero. As the compiler disables the interrupts by default in ISRs, there is no need to disable them explicitly in the code and 16-bit variables can be manipulated without any worry.
    if (max_counter < num_counters)                             // Keep track of how many counters have been used. When the average is cleared, we start with one counter and we increase this number every time until we reach num_counters, so that the average is always right.
        max_counter++;
    data_updated = DATA_READY;
}



// --------------------------------------------------------------------------
// For the SI-8B tube we have a nice curve to convert from cpm into uR/h, 
// so we don't need to compensate for dead time. This curve has been 
// digitized, converted into straight segments and represented in the 
// constants below. This procedure does a linear interpolation within those 
// segments and returns the correct result in uR/h.
// --------------------------------------------------------------------------
unsigned long int calculate_uR_h(unsigned long int cpm)
{
    unsigned long int cpm_to_sub[] =  { 381000, 364800, 347400, 327600, 303000, 271800,     // First: subtract the largest possible of these constants from your cpm.
                                        241800, 218400, 181200, 144000, 133800, 122400, 
                                        110400,  98400,  85800,  71400,  60000,  54240,
                                         48060,  41820,  35880,  30180,  24720,      0 };
    unsigned long int multiply_cpm[] = { 23556,  22222,  20977,  17424,  15406,  11314,     // Second: multiply by this corresponding constant (and divide by 10'000).
                                          7107,   6205,   5473,   4110,   3520,   3175, 
                                          3092,   2950,   2841,   2614,   2521,   2465, 
                                          2324,   2308,   2138,   2075,   2007,   1462 };
    unsigned long int uR_h_to_add[] = { 287800, 251800, 215300, 180800, 142900, 107600,     // Finally: add this corresponding constant to get uR/h.
                                         86280,  71760,  51400,  36110,  32520,  28900, 
                                         25190,  21650,  18070,  14440,  11460,  10040, 
                                          8604,   7164,   5894,   4711,   3615,      0 };
    char i;                                                     // This will be the index of the correct conversion factors.

    for (i=0; i<sizeof(cpm_to_sub); i++)                        // First find the correct segment of our interpolation curve.
    {
        if (cpm >= cpm_to_sub[i])
            break;
    }
    cpm = cpm - cpm_to_sub[i];                                  // Then perform the three steps of the conversion: subtract, multiply and add.
    cpm *= multiply_cpm[i];
    cpm /= 10000;
    cpm += uR_h_to_add[i];

    return(cpm);
}



// --------------------------------------------------------------------------
// This function returns the dose in nSv/h when fed with uR/h. 
// The conversion is just a unit change knowing that 1 mR/h is approximately 
// 8.74 uSv/h.
// --------------------------------------------------------------------------
unsigned long int calculate_nSv_h(unsigned long int uR_h)
{
    return((uR_h * 874) / 100);                                 // 1 mR/h is approximately 8.74 uSv/h.
}



// --------------------------------------------------------------------------
// This procedure will display the measured dose on the LCD screen. 
// According to the global variable "unit", the measurement can be displayed 
// in nano-Sieverts per hour, micro-Roentgen per hour or counts per minute. 
// It assumes that the global variables cpm_current, uR_h_current and 
// nSv_h_current already contain the correct values in their respective 
// units.
//
// Counts per minute:
// ------------------
// The result is always displayed as "xxxxxx cpm", leading zeros are 
// suppressed. The SI-8B will operate between in the 0..390'000 cpm range. 
//
// nano-Sieverts per hour:
// -----------------------
// With only 3 significant digits, results are displayed as: "x nSv/h",
// "xx nSv/h", "xxx nSv/h", "x.xx uSv/h", "xx.x uSv/h", "xxx uSv/h" and 
// "x.xx mSv/h". The SI-8B will operate between in the 0..2'520'000 nSv/h 
// range.
//
// micro-Roentgens per hour:
// -------------------------
// With only 3 significant digits, results are displayed as: "x uR/h",
// "xx uR/h", "xxx uR/h", "x.xx mR/h", "xx.x mR/h" and "xxx mR/h". The SI-8B 
// will operate between in the 0..288'000 cpm range.
//
// --------------------------------------------------------------------------
void display_dose()
{
    char str_value[10];
    char str_value_1[10];
    char str_unit[8];

    unsigned long int value_to_display;
    
    switch (unit_type)                                          // Start by selecting which value to display and prepare the corresponding unit.
    {
        case UNIT_R   : value_to_display = uR_h_current;
                        strcpy(str_unit,"uR/h");
                        if (value_to_display >= UR_H_RED)       // Also choose the color of the dose to display (green for low, orange for medium and red for high).
                            tft.setTextColor(TFT_RED, TFT_WHITE);
                        else if (value_to_display >= UR_H_ORANGE)
                            tft.setTextColor(TFT_ORANGE, TFT_WHITE);
                        else
                            tft.setTextColor(TFT_DARK_GREEN, TFT_WHITE);
                        break;
        case UNIT_SV  : value_to_display = nSv_h_current;
                        strcpy(str_unit,"nSv/h"); 
                        if (value_to_display >= NSV_H_RED)
                            tft.setTextColor(TFT_RED, TFT_WHITE);
                        else if (value_to_display >= NSV_H_ORANGE)
                            tft.setTextColor(TFT_ORANGE, TFT_WHITE);
                        else
                            tft.setTextColor(TFT_DARK_GREEN, TFT_WHITE);
                        break;
        default       : value_to_display = cpm_current;
                        strcpy(str_unit,"cpm"); 
                        if (value_to_display >= CPM_RED)
                            tft.setTextColor(TFT_RED, TFT_WHITE);
                        else if (value_to_display >= CPM_ORANGE)
                            tft.setTextColor(TFT_ORANGE, TFT_WHITE);
                        else
                            tft.setTextColor(TFT_DARK_GREEN, TFT_WHITE);
                        break;
                        break;
    }

    if ((value_to_display < 1000) || (unit_type == UNIT_CPM))   // Numbers from 0 to 999 and CPM of all sizes are printed as integers with no decimal point nor multiplier.
    {
        sprintf(str_value, "%lu", value_to_display);
    }
    else
    {
        if (value_to_display >= 1000000)                        // Numbers equals or larger than 1'000'000 are divided by 1'000 and the unit is corrected by two prefixes (a second division by 1'000 is performed below when computing the digits).
        {
            value_to_display /= 1000;
            switch (str_unit[0])
            {
                case 'p'  : str_unit[0] = 'u'; break;
                case 'n'  : str_unit[0] = 'm'; break;
                case 'u'  : str_unit[0] = ' '; break;
                case 'm'  : str_unit[0] = 'k'; break;
            }
        }
        else                                                    // For numbers in the 1000..999'999 range we only correct the multiplier prefix by one step to take into account the division by 1'000 performed below.
        {
            switch (str_unit[0])
            {
                case 'p'  : str_unit[0] = 'n'; break;
                case 'n'  : str_unit[0] = 'u'; break;
                case 'u'  : str_unit[0] = 'm'; break;
                case 'm'  : str_unit[0] = ' '; break;
            }
        }
        sprintf(str_value, "%lu", value_to_display / 1000);     // First compute the integer part.
        if (value_to_display < 100000)                          // No decimal point for numbers larger than 100'000, otherwise append a decimal point and compute the fractional part.
        {
            strcat(str_value, ".");
            sprintf(str_value_1, "%03lu", value_to_display % 1000); // We need to pad on 3 digits because of the division by 1'000.
            str_value_1[2] = 0;                                 // But we only want a maximum of two, so we shorten the string by terminating it with 0 one character before.
            if (value_to_display >= 10000)                      // Or just one decimal figure for numbers above or equal to 10'000, so in this case we shorten the string even more.
                str_value_1[1] = 0;
            strcat(str_value, str_value_1);                     // And we concatenate the integer and fractional parts together.
        }
    }

    while (strlen(str_value) <= 6)                              // To make sure that the new string will cover the old one, we add spaces to the left and to the right to have a string always longer than 6 characters.
    {
        strcat(str_value, " ");
        strcpy(str_value_1, " ");
        strcat(str_value_1, str_value);
        strcpy(str_value, str_value_1);
    }

    //tft.setTextColor(TFT_BLACK, TFT_WHITE);                     // Color was already selected before. Un-comment for having the dose always displayed in black.
    tft.setTextSize(5);                                         // And now we can display it on the TFT.
    draw_centre_string(str_value, 160, 160);

    strcpy(str_value_1, " ");                                   // To make sure that the new string will cover the old one, we add spaces to the left and to the right to have a string always longer than 5 characters.
    strcat(str_value_1, str_unit);
    strcpy(str_unit, str_value_1);
    strcat(str_unit, " ");
    tft.setTextSize(4);
    draw_centre_string(str_unit, 160, 200);
}



// --------------------------------------------------------------------------
// This procedure will display the analog gauge on the screen. It will take 
// the correct value to display from uR_h_current, nSv_h_current or 
// cpm_current, figure out the corresponding dose color thresholds and scale 
// the values to fit in the 0..100 range. Larger values are divided by 10 as 
// many times as required to fit in this range. Units are handled so that 
// meaningful figures are displayed on the scale. When the scale factor 
// changes the whole scale is redrawn with new figures and new colored arcs 
// to match the thresholds. As long as the factor is the same, the scale is 
// not redrawn (unless force_refresh is true). Figures are plotted outside 
// the gauge so that the needle doesn't cover them. The needle is actually 
// slightly shorter than the scale itself, so that it always has a white 
// background and can be redrawn almost flicker-free.
// --------------------------------------------------------------------------
void display_gauge(char force_refresh)
{
    long int value_to_display;                                  // Here we compute the value to display. Must be a signed variable to correctly compute coordinates.
    char multiplier;                                            // A variable used to count how many powers of 10 we shifted the value to display.
    static char old_multiplier = 255;                           // A variable to keep track of the old scale used to redraw the face of the gauge when the scale changes.
    int orange_thresold, red_thresold;                          // Red and orange dose thresholds that will be scaled down to match the gauge range.
    float x_sin, y_cos;                                         // Dummy variables to store the trigonometric projection of the line we are calculating.

    if (force_refresh)                                          // If force_refresh is true, the old multiplier is set to 255 to force redrawing the whole scale even if the multiplier did not change.
        old_multiplier = 255;

    switch (unit_type)                                          // Start by selecting which value to display and prepare the corresponding unit and scale.
    {
        case UNIT_R   : value_to_display = uR_h_current;
                        orange_thresold = UR_H_ORANGE;
                        red_thresold = UR_H_RED;
                        break;
        case UNIT_SV  : value_to_display = nSv_h_current;
                        orange_thresold = NSV_H_ORANGE;
                        red_thresold = NSV_H_RED;
                        break;
        default       : value_to_display = cpm_current;
                        orange_thresold = CPM_ORANGE;
                        red_thresold = CPM_RED;
                        break;
    }

    multiplier = 0;                                             // Start with no multiplier at all.
    while (value_to_display > 100)                              // Divide by 10 until we have something in the 0..100 range.
    {
        value_to_display /= 10;                                 // Scale the value to display.
        orange_thresold /= 10;                                  // Scale thresholds, too.
        red_thresold /= 10;
        multiplier++;                                           // And keep track of how many times we did this.
    }

    if (multiplier != old_multiplier)                           // If the new multiplier is different than the old one, we need to redraw the face of the display.
    {
        int x0, y0;                                             // Lowest coordinate of a tick.
        int x1, y1;                                             // Highest coordinate of a tick.
        int x2, y2;                                             // Lowest coordinate of the next tick on the right.
        int x3, y3;                                             // Highest coordinate of the next tick on the right.
        int x4, y4;                                             // Even higher coordinate of a long tick (it's above x1,y1).
        int i;                                                  // This is both a counter and the angle.

        old_multiplier = multiplier;                            // Keep track of the old multiplier to know if the scale needs to be redrawn.

        i = 0;                                                  // Calculate coordinates if the first tick mark.
        x_sin = sin((i - 50) * ANGLE * PI/180);
        y_cos = cos((i - 50) * ANGLE * PI/180);
        x0 = CENTER_X + RADIUS * x_sin;                         // x0,y0 are on the scale circumference.
        y0 = CENTER_Y - RADIUS * y_cos;
        x1 = CENTER_X + (RADIUS + TICK_LENGTH) * x_sin;         // x1,y1 are one short tick above x0,y0.
        y1 = CENTER_Y - (RADIUS + TICK_LENGTH) * y_cos;
        x4 = CENTER_X + (RADIUS + TICK_LENGTH_LONG) * x_sin;    // x4,y4 are one long tick above x0,y0.
        y4 = CENTER_Y - (RADIUS + TICK_LENGTH_LONG) * y_cos;

        do                                                      // Now compute all other ticks.
        {
            i += 5;                                             // A tick mark every 5 units.

            x_sin = sin((i - 50) * ANGLE * PI/180);             // Calculate coordinates of the next tick mark
            y_cos = cos((i - 50) * ANGLE * PI/180);
            x2 = CENTER_X + RADIUS * x_sin;                     // x2,y2 are on the scale circumference, one tick on the right than x0y0.
            y2 = CENTER_Y - RADIUS * y_cos;
            x3 = CENTER_X + (RADIUS + TICK_LENGTH) * x_sin;     // x3,y3 are one short tick above x2,y2.
            y3 = CENTER_Y - (RADIUS + TICK_LENGTH) * y_cos;

            if (i <= orange_thresold)                           // Fill area between marks with two triangles of the corresponding color.
            {
                tft.fillTriangle(x0, y0, x1, y1, x2, y2, TFT_GREEN);    // The trapezoidal shape is filled with two triangles.
                tft.fillTriangle(x1, y1, x2, y2, x3, y3, TFT_GREEN);
            }
            else if (i <= red_thresold)
            {
                tft.fillTriangle(x0, y0, x1, y1, x2, y2, TFT_ORANGE);
                tft.fillTriangle(x1, y1, x2, y2, x3, y3, TFT_ORANGE);
            }
            else
            {
                tft.fillTriangle(x0, y0, x1, y1, x2, y2, TFT_RED); 
                tft.fillTriangle(x1, y1, x2, y2, x3, y3, TFT_RED);
            }
                
            if ((i - 5) % 25 == 0)                              // Draw a tick mark. Multiples of 25 are longer than regular ticks. Beware that we already increased i by 5.
                tft.drawLine(x0, y0, x4, y4, TFT_BLACK);        // Long tick.
            else
                tft.drawLine(x0, y0, x1, y1, TFT_BLACK);        // Short tick.

            x0 = x2;                                            // Shift by one tick. 
            y0 = y2;
            x1 = x3; 
            y1 = y3;
            x4 = CENTER_X + (RADIUS + TICK_LENGTH_LONG) * x_sin;
            y4 = CENTER_Y - (RADIUS + TICK_LENGTH_LONG) * y_cos;
        }
        while (i < 100);                                        // Do this for the whole 0..100 range.
        tft.drawLine(x0, y0, x4, y4, TFT_BLACK);                // And don't forget to draw the last tick.

        tft.setTextSize(2);                                     // Now draw tick units.
        tft.setTextColor(TFT_BLACK, TFT_WHITE);
        draw_centre_string("0", 20, 55);                        // Zero is always "0", no matter the scale.
        switch (multiplier)
        {
            case 0:  
            case 3: draw_centre_string(" 25 ", 85, 35);
                    draw_centre_string(" 50 ", 160, 27);
                    draw_centre_string(" 75 ", 235, 35);
                    draw_right_string(" 100", 315, 52);
                    break;
            case 1: 
            case 4: draw_centre_string("250", 85, 35);
                    draw_centre_string("500", 160, 27);
                    draw_centre_string("750", 235, 35);
                    draw_right_string("1000", 315, 52);
                    break;
            case 2:
            case 5: draw_centre_string("2.5", 85, 35);
                    draw_centre_string("5.0", 160, 27);
                    draw_centre_string("7.5", 235, 35);
                    draw_right_string("  10", 315, 52);
                    break;
        }
    }

    // Plot needle.

    static int needle_x0 = 160;                                 // Static variables to keep track of the last needle position. x0,y0 id the point closer to the scale.
    static int needle_y0 = 120;                                 // And x1,y1 is the point closer to the bottom edge of the gauge.
    static int needle_x1 = 160;                                 // We just initialize them in center of the screen so that drawing a white line the first time doesn't overwrite anything.
    const int NEEDLE_Y1 = 150;                                  // The y1 coordinate of the needle is constant, as we want to keep the lower part of the display free for printing the dose.

    tft.drawLine(needle_x0, needle_y0, needle_x1, NEEDLE_Y1, TFT_WHITE);        // First delete the last needle by overwriting it with a white line.
    tft.drawLine(needle_x0+1, needle_y0, needle_x1+1, NEEDLE_Y1, TFT_WHITE);    // Doubling the line makes it thicker and easier to see.
    //tft.drawLine(needle_x0-1, needle_y0, needle_x1-1, NEEDLE_Y1, TFT_WHITE);

    x_sin = sin((value_to_display - 50) * ANGLE * PI/180);      // Compute the trigonometric projection of the needle angle.
    y_cos = cos((value_to_display - 50) * ANGLE * PI/180);
    needle_x0 = CENTER_X + (RADIUS-1) * x_sin;                  // x0,y0 are on the scale circumference. RADIUS is reduced by one to make sure we don't draw on the scale.
    needle_y0 = CENTER_Y - (RADIUS-1) * y_cos;
    needle_x1 = CENTER_X - ((NEEDLE_Y1 - CENTER_Y) / y_cos) * x_sin;    // x1 is calculated by finding the corresponding radius knowing the fixed coordinate y1.
    tft.drawLine(needle_x0, needle_y0, needle_x1, NEEDLE_Y1, TFT_BLACK);        // Draw the needle.
    tft.drawLine(needle_x0+1, needle_y0, needle_x1+1, NEEDLE_Y1, TFT_BLACK);    // And double it to make it more visible.
    //tft.drawLine(needle_x0-1, needle_y0, needle_x1-1, NEEDLE_Y1, TFT_BLACK);    // Tripling it is too thick.
}



// --------------------------------------------------------------------------
// The TFT library doesn't come with text alignment functions, so this 
// procedure will display a centered text at the corresponding screen 
// coordinates x and y. The text is only centered horizontally; the vertical 
// coordinate is not changed: it’s a center-top alignment. In order for this 
// to work, the text should fit in the screen: alignment will be wrong is 
// the text is larger than the screen width.
// --------------------------------------------------------------------------
void draw_centre_string(const char *buf, int x, int y)
{
    int16_t x1, y1;
    uint16_t w, h;
    tft.getTextBounds(buf, 0, 0, &x1, &y1, &w, &h);             // Get the size of the text.
    tft.setCursor(x-(w/2), y);                                  // Set cursor accordingly.
    tft.print(buf);                                             // Display it.
}



// --------------------------------------------------------------------------
// The TFT library doesn't come with text alignment functions, so this 
// procedure will display a right aligned text at the corresponding screen 
// coordinates x and y. only the horizontal coordinate is modified; the 
// vertical coordinate is not changed: it’s a right-top alignment. In order 
// for this to work, the text should fit in the screen: alignment will be 
// wrong is the text is larger than the screen width.
// --------------------------------------------------------------------------
void draw_right_string(const char *buf, int x, int y)
{
    int16_t x1, y1;
    uint16_t w, h;
    tft.getTextBounds(buf, 0, 0, &x1, &y1, &w, &h);             // Get the size of the text.
    tft.setCursor(x-w, y);                                      // Set cursor accordingly.
    tft.print(buf);                                             // Display it.
}



// --------------------------------------------------------------------------
// This procedure will display the integration time in seconds at the right 
// place on the screen.
// --------------------------------------------------------------------------
void display_integration_time(void)
{
    char str_value[7] = "";

    sprintf(str_value, "%d s ", time_interval);                 // Makes sure there is a space at the end to overwrite the old text.
    tft.setCursor(5, 5);
    tft.setTextColor(TFT_BLACK, TFT_WHITE);  
    tft.setTextSize(2);
    tft.print(str_value);
}



// --------------------------------------------------------------------------
// This procedure will return the battery voltage by reading the stepped 
// down voltage on the analog input pin VOLT_BATT.
// The voltage divider is composed by two resistors R1=100k and R2=18k, step 
// down factor is therefore 1:6.556. 
// The ADC converter in the Arduino has a resolution of ADC_M=1024 points 
// and a voltage reference U_ref=1.1V. Knowing the ADC raw reading ADC#, we 
// can calculate the rail voltage U_rail with the following relation:
//
// U_rail = ADC# * (U_ref / ADC_M) * ((R1 + R2) / R2)
//
// Voltage is returned in millivolts, but the resolution is only 
//
// resolution = (U_ref / ADC_M) * ((R1 + R2) / R2) = 7.04 mV
//
// It's important to use unsigned long int variables and specify constants 
// as "UL" to avoid rounding errors.
// --------------------------------------------------------------------------
unsigned int read_battery_voltage(void)
{
    unsigned long adc_value;

    adc_value = analogRead(VOLT_BATT);                          // Read the voltage on VOLT_IN.

    adc_value = adc_value * 1100UL * (100UL + 18UL);            // Calculate rail voltage with the equation explained above.
    adc_value = adc_value / (1024UL * 18UL);
    return(adc_value);
}



// --------------------------------------------------------------------------
// This procedure will display on the screen the specified voltage.
// The variable is expressed in millivolts and it will be written in the
// "x.xx V" form, unit included. 
// It will flash red if the voltage is outside the good battery voltage 
// range.
// If the voltage is close to 0, the instrument is powered via the USB port
// and this procedure will display "USB" instead of the voltage.
// --------------------------------------------------------------------------
void display_voltage(unsigned int v_batt)
{
    static byte blink_counter = 0;                              // Counter used to slow down the blinking of the empty battery.

    char str_value[7] = "";
    char str_value_1[4] = "";

    if (v_batt < VBATT_MIN)                                     // When powered by the USB port, the battery will read 0V.
    {
        tft.setTextColor(TFT_BLACK, TFT_WHITE);                 // If it's the case, display "USB" instead of "0.00 V".
        tft.setTextSize(2);
        draw_right_string("   USB", 315, 5);
        return;
    }

    sprintf(str_value, " %d", v_batt / 1000);                   // Add a space in front of the digit to always clear the text (text is right justified).
    strcat(str_value, ".");
    sprintf(str_value_1, "%02d", (v_batt % 1000) / 10);
    strcat(str_value, str_value_1);
    strcat(str_value, " V");

    if (v_batt < VBATT_ZERO)                                    // Check if the voltage is low.
    {
        if (blink_counter < 5)                                  // Cycle between the two colors if it's the case.
            tft.setTextColor(TFT_RED, TFT_WHITE);  
        else
            tft.setTextColor(TFT_BLACK, TFT_WHITE);  
        blink_counter++;
        if (blink_counter > 10)
            blink_counter = 0;
    }
    else
    {
        tft.setTextColor(TFT_BLACK, TFT_WHITE);                 // Battery is ok, stick to black.
    }  
    tft.setTextSize(2);
    draw_right_string(str_value, 315, 5);
}



// --------------------------------------------------------------------------
// This procedure displays the status of the speaker in the the bottom left
// corner of the screen.
// --------------------------------------------------------------------------
void display_sound_status(void)
{
    tft.setCursor(5, 220);
    tft.setTextSize(2);
    if (tube_ticks == TICKS_ENABLED)
    {
        tft.setTextColor(TFT_BLACK, TFT_WHITE);
        tft.print("(( )) ");
    }
    else
    {
        tft.setTextColor(TFT_LIGHT_GRAY, TFT_WHITE);
        tft.print("((X)) ");
    }
}



// --------------------------------------------------------------------------
// This procedure will set the brightness of the TFT screen according to 
// display_brightness.
// --------------------------------------------------------------------------
void set_tft_brightness(void)
{
    if (display_brightness == DISPLAY_BRIGHT)                   
        digitalWrite(LCD_LED, HIGH);                            // Fully on for a bright display.
    else
        analogWrite(LCD_LED, DISPLAY_DIM_VALUE);                // Or "PWMed down" for a dim display.
}



// --------------------------------------------------------------------------
// This procedure only displays my initials in the bottom right corner of 
// the screen.
// --------------------------------------------------------------------------
void display_my_initials(void)
{
    tft.setTextSize(2);
    tft.setTextColor(TFT_BLACK, TFT_WHITE);
    draw_right_string("I.G.", 315, 220);
}



// --------------------------------------------------------------------------
// This procedure will display the welcome screen on the LCD display and also
// send it on the serial port.
// This procedure is blocking: while this procedure is running (the about 
// screen is displayed) no dose is computed and no values are sent over the 
// serial port. I think this is ok for an about screen. 
// If wait_for_touch is true, it will wait until the screen is touched or
// for 10 seconds. If it's false it will just wait 1 second.
// --------------------------------------------------------------------------
void display_welcome_screen(bool wait_for_touch)
{
    const int TREFOIL_CENTRE_X = 160;
    const int TREFOIL_CENTRE_Y = 120;
    const int TREFOIL_RADIUS = 50;
    int i;
    float x_sin, y_cos;                                         // Dummy valuables to store the trigonometric projection of the line we are calculating.
    int x0, y0, x1, y1;
    int x2, y2, x3, y3;

    tft.fillScreen(TFT_WHITE);                                  // Start by clearing the screen.
    set_tft_brightness();                                       // And make sure the back-light is lit (when the instrument boots up, the back-light is kept off until this point to avoid flicker).

    tft.setTextColor(TFT_BLACK);                                // Print the welcome text lines.
    tft.setTextSize(2);
    draw_centre_string(WELCOME_MSG_1, 160, 10);
    draw_centre_string(WELCOME_MSG_2, 160, 30);
    draw_centre_string(WELCOME_MSG_3, 160, 195);
    draw_centre_string(WELCOME_MSG_4, 160, 215);

    // Now draw the radioactive trefoil. It would be easier to directly display a bitmap, but this eats up too much flash memory.

    tft.fillRect(TREFOIL_CENTRE_X - (TREFOIL_RADIUS * 1.1), TREFOIL_CENTRE_Y - (TREFOIL_RADIUS * 1.1), TREFOIL_RADIUS * 2.2, TREFOIL_RADIUS * 2.2, TFT_YELLOW);
    tft.fillCircle(TREFOIL_CENTRE_X, TREFOIL_CENTRE_Y, TREFOIL_RADIUS / 5, TFT_BLACK);
    x0 = 0;
    y0 = 0;
    x1 = 0;
    y1 = 0;
    for (i=0; i<360; i+=15)
    {
        x_sin = sin(i * PI/180);                                // Calculate coordinates of the next sector
        y_cos = cos(i * PI/180);
        x2 = TREFOIL_CENTRE_X + TREFOIL_RADIUS * x_sin;         // x0,y0 are on the outer circumference.
        y2 = TREFOIL_CENTRE_Y - TREFOIL_RADIUS * y_cos;
        x3 = TREFOIL_CENTRE_X + TREFOIL_RADIUS * x_sin * 1.5 / 5;   // x1,y1 are on the inner circumference.
        y3 = TREFOIL_CENTRE_Y - TREFOIL_RADIUS * y_cos * 1.5 / 5;

        if (((i > 30) && (i <= 90)) || ((i > 150) && (i <= 210)) || ((i > 270) && (i <= 330)))
        {
            tft.fillTriangle(x0, y0, x1, y1, x2, y2, TFT_BLACK);
            tft.fillTriangle(x1, y1, x2, y2, x3, y3, TFT_BLACK);
        }

        x0 = x2;
        y0 = y2;
        x1 = x3;
        y1 = y3;
    }

    Serial.print(WELCOME_MSG_1);                                // Print welcome message also on the serial port.
    Serial.print(", ");
    Serial.print(WELCOME_MSG_2);
    Serial.print(", ");
    Serial.print(WELCOME_MSG_3);
    Serial.print(", ");
    Serial.print(WELCOME_MSG_4);
    Serial.println();
    Serial.println();

    if (wait_for_touch)
    {
        for (i = 0; i < 1000; i++)                              // Wait some time (5 seconds) or until the user touches the screen.
        {
            delay(10);
            if (ts.touched())                                   // Break the loop if the screen is touched.
            {
                delay(100);                                     // De-bounce, sort of.
                break;
            }
        }
    }
    else
    {
        delay(1000);                                            // Simply wait 1 second.
    }

    tft.fillScreen(TFT_WHITE);                                  // Clear the screen again.

    Serial.println(F("Time period [ms];Raw counts [1];Counts per minute [cpm];Dose [uR/h];Dose [nSv/h];Average size [1];Battery voltage [mV]"));    // Display the data header on the serial port.
}



// --------------------------------------------------------------------------
// This procedure will take care of the touch panel and is executed every 
// main loop.
// --------------------------------------------------------------------------
void touch_panel_handler(void)
{
    char i;
    int ts_x, ts_y;

    if (ts.touched())
    {
        TS_Point p = ts.getPoint();
        ts_x = ((long int)p.x - TS_MIN_X) * 320 / (TS_MAX_X - TS_MIN_X);
        ts_x = 320 - ts_x;
        ts_y = ((long int)p.y - TS_MIN_Y) * 240 / (TS_MAX_Y - TS_MIN_Y);
        //Serial.print(F("ts_x = ")); Serial.print(p.x); Serial.print(F(", ts_y = ")); Serial.print(p.y); Serial.print(F(", scr_x = ")); Serial.print(ts_x); Serial.print(F(", scr_y = ")); Serial.println(ts_y);

        //tft.drawRect(80, 190, 240-80-1, 240-190-1, TFT_YELLOW);     // Un-comment to visualize the area on the screen.
        if ((ts_x > 80) && (ts_x < 240) &&
            (ts_y > 190) && (ts_y < 240))                       // Check if the click was in the "unit" area.
        {
            switch (unit_type)                                  // Valid click: cycle between available units. 
            {
                case UNIT_CPM  : unit_type = UNIT_R; break;
                case UNIT_R    : unit_type = UNIT_SV; break;
                case UNIT_SV   : unit_type = UNIT_CPM; break;
                default        : unit_type = UNIT_CPM; break;
            }
            eeprom_write_config();                              // Save new value in the EEPROM.
            display_dose();                                     // Update the display with the new unit.
        }

        //tft.drawRect(0, 0, 80-0-1, 35-0-1, TFT_YELLOW);             // Un-comment to visualize the area on the screen.
        if ((ts_x > 0) && (ts_x < 80) &&
            (ts_y > 0) && (ts_y < 35))                          // Check if the click was in the "time" area.
        {
            switch (time_interval)                              // Valid click: cycle between integration times. 
            {
                case TIME_SLOW : time_interval = TIME_MED; break;
                case TIME_MED  : time_interval = TIME_FAST; break;
                case TIME_FAST : time_interval = TIME_SLOW; break;
                default        : time_interval = TIME_MED; break;
            }
            num_counters = (TIMER_PERIOD * time_interval / 1000000L) + 1;   // Update number of counters in use and clear all counters.
            noInterrupts();                                     // Interrupts should be disabled while resetting the counters.
            for (i=0; i<NUM_COUNTERS; i++)
                counters[i] = 0;
            current_counter = 0;
            max_counter = 1;
            interrupts();
            display_integration_time();                         // Update display with the new time.
            eeprom_write_config();                              // Save new value in the EEPROM.
        }

        //tft.drawRect(0, 205, 80-0-1, 240-205-1, TFT_YELLOW);    // Un-comment to visualize the area on the screen.
        if ((ts_x > 0) && (ts_x < 80) &&
            (ts_y > 205) && (ts_y < 240))                       // Check if the click was in the "sound" area.
        {
            if (tube_ticks == TICKS_ENABLED)                    // Invert tube_ticks and set TICK_EN accordingly.
            {
                tube_ticks = TICKS_DISABLED;
                digitalWrite(TICK_EN, LOW);
            }
            else
            {
                tube_ticks = TICKS_ENABLED;
                digitalWrite(TICK_EN, HIGH);
            }
            display_sound_status();                             // Update the display with the new speaker status.
            eeprom_write_config();                              // Save new value in the EEPROM.
        }

        //tft.drawRect(240, 0, 320-240-1, 35-0-1, TFT_YELLOW);    // Un-comment to visualize the area on the screen.
        if ((ts_x > 240) && (ts_x < 320) &&
            (ts_y > 0) && (ts_y < 35))                          // Check if the click was in the "voltage" area.
        {
            if (display_brightness == DISPLAY_BRIGHT)           // Toggle the brightness status
                display_brightness = DISPLAY_DIM;
            else
                display_brightness = DISPLAY_BRIGHT;
            set_tft_brightness();                               // And set it.
            eeprom_write_config();                              // Save new value in the EEPROM.
        }

        //tft.drawRect(240, 205, 320-240-1, 240-205-1, TFT_YELLOW);    // Un-comment to visualize the area on the screen.
        if ((ts_x > 240) && (ts_x < 320) &&
            (ts_y > 205) && (ts_y < 240))                       // Check if the click was in the "initials" area.
        {
            display_welcome_screen(true);                       // Display "about" screen and wait for the touch screen 

            display_dose();                                     // Refresh display
            display_gauge(true);                                // Force a scale redraw
            display_integration_time();
            display_sound_status();
            display_my_initials();
            display_voltage(volt_batt);    
        }
    }
}



// --------------------------------------------------------------------------
// This function will read the configuration variables from the EEPROM at 
// their corresponding addresses.
// Because there is no easy way to load values in the EEPROM from the C 
// source with the Arduino compiler as we flash it, we use one byte of the 
// EEPROM as a flag to see if the EEPROM has already been flashed once or 
// not.
// We read the byte at the EE_ADDR_EEINIT address. If the EEPROM is empty, it 
// will most probably return 0x00 or 0xFF. If we don't read back the specific 
// value defined in EE_INIT_VALUE we know that the EEPROM has never been 
// programmed and we write our configuration variables to the EEPROM. So, the
// values put in the variables at initialization have meaningful data, are 
// effectively saved to the EEPROM and will be still the same the next time 
// the Arduino boots.
// We also write EE_INIT_VALUE at EE_ADDR_EEINIT, so this only happens the 
// first time ever we run the code and from this moment on the data will not 
// be overwritten.
// Any value different from 0x00 and 0xFF will work for EE_INIT_VALUE. And to 
// refresh the EEPROM with new values in the source code, simply change it to
// a different value.
// --------------------------------------------------------------------------
void eeprom_read_config(void)
{
    if (EEPROM.read(EE_ADDR_EEINIT) != EE_INIT_VALUE)           // Check if the EEPROM has already been written with our configuration once in the Arduino lifetime.
    {
        eeprom_write_config();                                  // If not, simply write the configuration variables as they have been initialized by this source code.
        EEPROM.write(EE_ADDR_EEINIT, EE_INIT_VALUE);            // And set the flag showing that the EEPROM has been programmed, so that it wont be rewritten with default values at boot time.
        return;
    }

    time_interval = EEPROM.read(EE_ADDR_TIME_INT);              // If the EEPROM contains a saved configuration, simply read it back and store in our configuration variables.
    unit_type = EEPROM.read(EE_ADDR_UNIT_TYPE);
    tube_ticks = EEPROM.read(EE_ADDR_TUBE_TKS);
    display_brightness = EEPROM.read(EE_ADDR_DISP_BRT);
}



// --------------------------------------------------------------------------
// This function will write the configuration variables in the EEPROM at 
// their corresponding addresses.
// --------------------------------------------------------------------------
void eeprom_write_config(void)
{
    EEPROM.write(EE_ADDR_TIME_INT, time_interval);
    EEPROM.write(EE_ADDR_UNIT_TYPE, unit_type);
    EEPROM.write(EE_ADDR_TUBE_TKS, tube_ticks);
    EEPROM.write(EE_ADDR_DISP_BRT, display_brightness);
}
